/*

Copyright (c) 2013 Centre for Water Systems,
                   University of Exeter

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/

#ifndef _CA_TABLE_HPP_
#define _CA_TABLE_HPP_


//! \file Table.hpp
//! Define table data, i.e. a constant array.
//! \author Michele Guidolin, University of Exeter, 
//! contact: m.guidolin [at] exeter.ac.uk
//! \date 2013-11


#include"Grid.hpp"
#include<limits>
#include<iostream>
#include<cstdlib>
#include"caapi2D.hpp"


namespace CA {
  

  //! Define a class that manage the data as in a table format, i.e. a
  //! constant array where a value can be identified by index. The
  //! array is constant during a CA function execution. It can/must be
  //! populated in the main code.
  //! \attention the index in the table is different than a CA_INDEX.
  template<typename T>  
  class Table: public CA::Uncopyable
  {
  public:

    //! Create the table data. It is possible to have implementation
    //! specific options set by using the options list. \attention Do
    //! not destroy the grid before destroying this buffer.
    //! \param grid        The Grid
    //! \param num         The number of elements.
    //! \param options     The list of implementation specific options.
    Table(Grid& grid, Unsigned num, const Options& options = Options());


    //! Destroy the table.
    virtual ~Table();


    //! Return the specific options about the Table object and this
    //! implementation.
    static Options options();


    //! Update the specific values of the table from the data of the
    //! given memory location.  The specific value are identified by a
    //! starting location and a end location. The memory is identified
    //! by a size.
    //! \param start         The startiong position where to update table.
    //! \param stop          The ending position where to update table (not included).
    //! \param mem           The pointer to the memory location where to read data.
    //! \param mem_num       The number of elements in memory location.
    void update(Unsigned start, Unsigned stop, const T* mem, Unsigned mem_num);


    // ---- DEBUG ----
      
    //! Dump the buffer (with borders) into the given output stream in a
    //! readable form. The method print the buffer starting from the
    //! top/left corner.
    //! \param out         The output stream/
    //! \param x_sep       The separator of values in the X dimension.
    //! \param y_sep       The separator of values in the Y dimension.
    void dump(std::ostream& out, String x_sep=" ", String y_sep="\n");    


    // ---------  Implementation dependent --------------


    //! Return the event that was generated by the last command that
    //! used (or is using) the table.
    cl::Event event() const;


    //! Set the event that was generated by the last command.
    void setEvent(cl::Event& event);


    //! Return the OpenCl buffer used by the table.
    cl::Buffer buffer() const;

    
  private:

    //! The reference to the grid.
    Grid& _grid;
    
    //! The number of elements in the openCL buffer that contains the table.
    _caUnsigned    _buff_num;


    //! The size of the OnpeCL buffer in bytes.
    _caUnsigned    _buff_size;


    //! OpenCL buffer with the values data.
    cl::Buffer     _buff;

    //! The last event generated by a command that was using this
    //! buffer. This can be used to synchonyse different command.
    cl::Event  _event;
  };


  /// ----- Inline implementation ----- ///

  
  template<typename T>
  inline Table<T>::Table(Grid& grid, Unsigned num, const Options& options):
    _grid(grid),
    _buff_num(std::max(static_cast<Unsigned>(1),num)),
    _buff_size(sizeof(T) * _buff_num),
    _buff(cl::Buffer(_grid.context(), CL_MEM_READ_ONLY,_buff_size)),
    _event()
  {    
    // The buffer of the table is created directly in the device
    // memory. It is expeted to be only read by the kernel.

    // Make sure the buffer is initialised to zero.
#ifdef  CA_OCL_USE_EVENTS 
    _grid.fill1DBuff(T(),_buff,0,_buff_num,NULL,&_event);
#else
    _grid.fill1DBuff(T(),_buff,0,_buff_num,NULL,NULL);
#endif
  }
  

  template<typename T>
  inline Table<T>::~Table()
  {
  }


  template<typename T>
  inline Options Table<T>::options()
  {
    Options options;

    // Add the optional arguments. The tag start from higher number
    // for safety reason.
    CA::Unsigned na = 6000;
    
    // Example
    //options.push_back(new Arguments::Arg(na++,"option-name", "Option desctiption","value",true, true, false));

    return options;
  }


  template<typename T>
  inline void Table<T>::update(Unsigned start, Unsigned stop, const T* mem, Unsigned mem_num)
  {
    if(std::min(stop-start,mem_num)<=0)
      return;

    Unsigned mem_size = std::min(std::min(stop-start,mem_num)*sizeof(T), _buff_size);

#ifdef  CA_OCL_USE_EVENTS 
    // Create the list of event t0 wait.
    std::vector<cl::Event> wait_events(1,_event);   

    // Write the data into the buffer non blocking.
    _grid.queue().enqueueWriteBuffer(_buff, CL_FALSE, start*sizeof(T), mem_size, mem,  &wait_events, &_event);   
#else
    _grid.queue().enqueueWriteBuffer(_buff, CL_FALSE, start*sizeof(T), mem_size, mem,  NULL, NULL);   
#endif
  }

  template<typename T>
  inline void Table<T>::dump(std::ostream& out, String x_sep, String y_sep)
  {
    // Create a temporary array to keep the buffer.
    std::vector<T> mem(static_cast<size_t>( _buff_num ));

#ifdef  CA_OCL_USE_EVENTS 
    // Wait for the last command issued that uses this bufer to
    // finish.
    _event.wait();

    // Copy the buffer to memory (blocking) abd keep store the event.
    _grid.queue().enqueueReadBuffer(_buff, CL_TRUE, 0, static_cast<size_t>(_buff_size), &mem[0], 0, &_event);   

#else    
    _grid.queue().enqueueReadBuffer(_buff, CL_TRUE, 0, static_cast<size_t>(_buff_size), &mem[0], 0, NULL);   
#endif    

    // Cycle through the table values
    for(_caUnsigned i=0; i<_buff_num; ++i)
    {
      out<<mem[static_cast<size_t>(i)];	
      out<<x_sep;
    }
    out<<y_sep;
    

#ifdef  CA_OCL_USE_EVENTS 
    // Set the event of this buffer to wait for all the previous
    // events (copy) to finish.
    _grid.queue().enqueueMarker(&_event);
#endif
  }


  template<typename T>
  inline cl::Event Table<T>::event() const
  {
    return _event;
  }



  template<typename T>
  inline void Table<T>::setEvent(cl::Event& event)
  {
    _event = event;
  }


  template<typename T>
  inline cl::Buffer Table<T>::buffer() const
  {
    return _buff;
  }


}



#endif	// _CA_TABLE_HPP_
