/*******************************/
/* Last updated on 25-Mar-2014 */
/* Written by Mr. Qi Wang      */
/* q.wang.pomodoro@gmail.com   */
/* Centre for Water Systems    */
/* University of Exeter        */
/*******************************/

#include <math.h>
#include "mex.h"
#include "epanet2mex.h"

void TLN2obj(double *vars, double *objs, double *constrs, int NFE) {
	
	float pipeCost, totalCost, networkResilience;
	int errorCode, i, j, k;
	float retrievedData, juncDemand[6], juncHead[6], juncPressure[6], juncU[6], reservoirDischarge[1], reservoirHead[1], pipeDiameter[8];
	float minHead[6], minPressure, pressureViolation;
	float diameterList[14]={1.0, 2.0, 3.0, 4.0, 6.0, 8.0, 10.0, 12.0, 14.0, 16.0, 18.0, 20.0, 22.0, 24.0};
	float unitCost[14]={2.0, 5.0, 8.0 ,11.0, 16.0, 23.0, 32.0, 50.0, 60.0, 90.0, 130.0, 170.0, 300.0, 550.0};
	int pipeSetting[8];
	int fromNode[8], toNode[8];
	int nPipe, pipeIndex[8];
	float sumDiameter, maxDiameter;
	float X, Xmax;
	static int numOfEvaluation=0;
	objs[0] = 0.0;
	objs[1] = 0.0;
	constrs[0] = 0.0;

	numOfEvaluation+=1;
	// open the hydraulic solver
	if (numOfEvaluation==1)
	{
		ENopen("TLN.inp","reportFile.rpt","");
		ENresetreport();
		ENsetreport("MESSAGES NO");
		printf("The INP file has been opened.\n");
	}		
	
	// initialisation
	errorCode=-1;
	minPressure=30.0;
	pressureViolation=0.0;
	for (i=0; i<8; i++)
	{
		pipeSetting[i]=floor(vars[i]);
	}
	
	// convert the setting of pipe diameter
	for (i=0; i<8; i++)
	{
		pipeDiameter[i]=25.4*diameterList[pipeSetting[i]]; // in. => mm
	}

	// change the diameter settings for pipes
	for (i=0; i<8; i++)
	{
		ENsetlinkvalue(i+1, EN_DIAMETER, pipeDiameter[i]);
	}
	
	// retrieve the static values of the network
	// pipe diameter & length
	totalCost=0;
	networkResilience=-1;
	for (i=0; i<8; i++)
	{
		ENgetlinkvalue(i+1, EN_LENGTH, &retrievedData);
		pipeCost=unitCost[pipeSetting[i]]*retrievedData;
		totalCost+=pipeCost;
	}
	
	// calculate the uniformity of the network
	for (i=0; i<8; i++)
	{
		ENgetlinknodes(i+1, &fromNode[i], &toNode[i]);
	}
	for (i=1; i<7; i++)
	{
		nPipe=0;
		maxDiameter=0;
		sumDiameter=0;
		for (j=0; j<8; j++)
		{
			if (fromNode[j]==i)
			{
				nPipe+=1;
				pipeIndex[nPipe-1]=j+1;
			}
			if (toNode[j]==i)
			{
				nPipe+=1;
				pipeIndex[nPipe-1]=j+1;
			}
		}
		for (k=0; k<nPipe; k++)
		{
			ENgetlinkvalue(pipeIndex[k], EN_DIAMETER, &retrievedData);
			sumDiameter+=retrievedData;
			if (retrievedData>maxDiameter)
			{
				maxDiameter=retrievedData;
			}
		}
		juncU[i-1]=sumDiameter/(nPipe*maxDiameter);
	}
	
	// calculate the minimum head requirement of each node
	for (i=0; i<6; i++)
	{
		ENgetnodevalue(i+1, EN_ELEVATION, &retrievedData);
		minHead[i]=retrievedData+minPressure;
	}
	
	// run the hydraulic simulation
	errorCode=ENsolveH();

	for (i=0; i<6; i++)
	{
		ENgetnodevalue(i+1, EN_DEMAND, &retrievedData);
		juncDemand[i]=retrievedData;
		ENgetnodevalue(i+1, EN_HEAD, &retrievedData);
		juncHead[i]=retrievedData;
		ENgetnodevalue(i+1, EN_PRESSURE, &retrievedData);
		juncPressure[i]=retrievedData;
		if (juncPressure[i]<minPressure)
		{
			pressureViolation+=(minPressure-juncPressure[i]);
		}
	}
	ENgetnodevalue(6+1, EN_DEMAND, &retrievedData);
	reservoirDischarge[0]=retrievedData;
	ENgetnodevalue(6+1, EN_HEAD, &retrievedData);
	reservoirHead[0]=retrievedData;
	
	// compute the objective functions
	X=0.0;
	Xmax=0.0;
	for (i=0; i<6; i++)
	{
		X+=juncU[i]*juncDemand[i]*(juncHead[i]-minHead[i]);
		Xmax+=juncDemand[i]*minHead[i];
	}

	Xmax=-1*reservoirDischarge[0]*reservoirHead[0]-Xmax;
	networkResilience=X/Xmax;

	// close the hydraulic solver
	if (numOfEvaluation==NFE)
	{
		ENclose();
		printf("The INP file has been closed.\n");
		numOfEvaluation=0;
	}
	
	// report the objectives and constraint violation
    objs[0] = totalCost/1000000.0; // $ => $ MM
    objs[1] = -networkResilience; // for minimisation purpose
	constrs[0] = -(pressureViolation+errorCode);
	return;
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
	double *vars,*objs,*constrs;
	int NFE;

	/* Check for proper number of arguments. */
	if(nrhs!=2)
	{
		mexErrMsgIdAndTxt( "MATLAB:TLN2obj:invalidNumInputs", "Two inputs required...");
	}
	else if(nlhs>2)
	{
		mexErrMsgIdAndTxt( "MATLAB:TLN2obj:maxlhs", "Too many output arguments...");
	}
    
  /* Create matrix for the return argument. */
  plhs[0] = mxCreateDoubleMatrix((mwSize)2, (mwSize)1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix((mwSize)1, (mwSize)1, mxREAL);
  
  /* Assign pointers to each input and output. */
  vars = mxGetPr(prhs[0]);
  NFE = mxGetScalar(prhs[1]);
  objs = mxGetPr(plhs[0]);
  constrs = mxGetPr(plhs[1]);
  
  /* Call the TLN2obj subroutine. */
  TLN2obj(vars, objs, constrs, NFE);
}